/*
 *  ======== TMP117.c ========
 *  TMP117 APIs for initialization and use of the TMP117 peripheral
 *
 */

#include <stddef.h>
#include <stdint.h>
#include <stdbool.h>

#include "inc/hw_memmap.h"

#include "driverlib/i2c.h"
#include "driverlib/sysctl.h"
#include "FreeRTOSConfig.h"
#include "TMP117.h"

#define MSB(u16) (((u16) & 0xFF00U) >> 8)
#define LSB(u16) ((u16) & 0xFFU)

/*
 *  ======== I2C Transfer ========
 *  TM4C-specific I2C send or receive function
 */
int8_t mcu_i2cTransfer( uint8_t busId, uint8_t i2cAddr,
                        uint8_t *dataToWrite, uint8_t writeLength,
                        uint8_t *dataToRead,  uint8_t readLength)
{
    if(dataToWrite != NULL)
    {
        /* Set up the I2C address with write transaction. */
        I2CMasterSlaveAddrSet(I2C1_BASE, i2cAddr, false);

        if (writeLength == 1)
        {
            /* Store the command data in I2C data register. */
            I2CMasterDataPut(I2C1_BASE, dataToWrite[0]);

            I2CMasterControl(I2C1_BASE, I2C_MASTER_CMD_SINGLE_SEND);

            /* Wait until the I2C transaction is complete. */
            while(I2CMasterBusy(I2C1_BASE))
            {
            }
        }
        if (writeLength == 3)
        {
            /* Store the command data in I2C data register. */
            I2CMasterDataPut(I2C1_BASE, dataToWrite[0]);

            /* Start the I2C transaction. */
            I2CMasterControl(I2C1_BASE, I2C_MASTER_CMD_BURST_SEND_START);

            /* Wait until the I2C transaction is complete. */
            while(I2CMasterBusy(I2C1_BASE))
            {
            }

            /* Store the command data in I2C data register. */
            I2CMasterDataPut(I2C1_BASE, dataToWrite[1]);

            /* Start the I2C transaction. */
            I2CMasterControl(I2C1_BASE, I2C_MASTER_CMD_BURST_SEND_CONT);

            /* Wait until the I2C transaction is complete. */
            while(I2CMasterBusy(I2C1_BASE))
            {
            }

            /* Store the command data in I2C data register. */
            I2CMasterDataPut(I2C1_BASE, dataToWrite[2]);

            /* Start the I2C transaction. */
            I2CMasterControl(I2C1_BASE, I2C_MASTER_CMD_BURST_SEND_FINISH);

            /* Wait until the I2C transaction is complete. */
            while(I2CMasterBusy(I2C1_BASE))
            {
            }
        }
    }

    if(dataToRead != NULL)
    {
        /* Modify the data direction to true, so that seeing the address will
         * indicate that the I2C Master is initiating a read from the TMP117.
         */
        I2CMasterSlaveAddrSet(I2C1_BASE, i2cAddr, true);

        if(readLength == 2)
        {
            /* Setup for first read.  Use I2C_MASTER_CMD_BURST_RECEIVE_START
             * to start a burst mode read.  The I2C master continues to own
             * the bus at the end of this transaction. */
            I2CMasterControl(I2C1_BASE, I2C_MASTER_CMD_BURST_RECEIVE_START);

            /* Wait until master module is done transferring.
             * The I2C module has a delay in setting the Busy flag in the
             * register so there needs to be a delay before checking the Busy
             * bit.  The below loops wait until the Busy flag is set, and then
             * wait until it is cleared to indicate that the transaction is
             * complete.  This can take up to 633 CPU cycles @ 100 kbit I2C
             * Baud Rate and 120 MHz System Clock.  Therefore, a while loop is
             * used instead of SysCtlDelay. */
            while(!I2CMasterBusy(I2C1_BASE))
            {
            }
            while(I2CMasterBusy(I2C1_BASE))
            {
            }

            /* Read the first byte data from the TMP117. */
            dataToRead[0] = I2CMasterDataGet(I2C1_BASE);

            /* Setup for the second read.  Use I2C_MASTER_CMD_BURST_RECEIVE_FINISH
             * to terminate the I2C transaction.  At the end of this transaction,
             * the STOP bit will be issued and the I2C bus is returned to the
             * Idle state. */
            I2CMasterControl(I2C1_BASE, I2C_MASTER_CMD_BURST_RECEIVE_FINISH);

            /* Wait until master module is done transferring. */
            while(!I2CMasterBusy(I2C1_BASE))
            {
            }
            while(I2CMasterBusy(I2C1_BASE))
            {
            }

            /* Read the second byte data from the TMP117. */
            dataToRead[1] = I2CMasterDataGet(I2C1_BASE);
        }
    }

    return (0);
}

/*
 *  ======== ms Delay ========
 *  TM4C-specific millisecond delay
 */
void mcu_msWait(unsigned long msWait)
{
    /*
     *  Add MCU specific wait loop for msWait. The unit is in milli-seconds
     */
    SysCtlDelay(((configCPU_CLOCK_HZ / 1000) / 3) * msWait);
}

/*
 *  ======== lock ========
 *  Lock out writes to EEPROM NVM
 */
static inline void lock(TMP117_Handle sensor)
{
    uint8_t txBuf[3];

    /* EEPROM_Unlock Register Write to Lock the NVM */
    txBuf[0] = TMP117_EEPROM;
    txBuf[1] = MSB(TMP117_EEPROM_EUN_DISABLE);
    txBuf[2] = LSB(TMP117_EEPROM_EUN_DISABLE);
    mcu_i2cTransfer(sensor->busId, sensor->devAddr, txBuf, 3, NULL, 0);
}

/*
 *  ======== unlock ========
 *  Unlock EEPROM NVM so sensor register writes will be saved
 */
static inline void unlock(TMP117_Handle sensor)
{
    uint8_t txBuf[3];

    /* EEPROM_Unlock Register Write to Unlock the NVM */
    txBuf[0] = TMP117_EEPROM;
    txBuf[1] = MSB(TMP117_EEPROM_EUN_ENABLE);
    txBuf[2] = LSB(TMP117_EEPROM_EUN_ENABLE);
    mcu_i2cTransfer(sensor->busId, sensor->devAddr, txBuf, 3, NULL, 0);
}

/*
 *  ======== pollBusy ========
 *  Poll EEPROM_Busy flag until it's 0 (not busy)
 */
static void pollBusy(TMP117_Handle sensor)
{
    uint8_t txBuf[1];
    uint8_t rxBuf[2];

    /* verify that EEPROM isn't busy */
    rxBuf[0] = MSB(TMP117_EEPROM_EEPROM_BUSY_ENABLE);
    do {
        /* EEPROM_Unlock Register Read to get the Busy flag */
        txBuf[0] = TMP117_EEPROM;
        mcu_i2cTransfer(sensor->busId, sensor->devAddr, txBuf, 1, rxBuf, 2);

        /* EEPROM_Busy is in the MSB of the unlock register */
    } while (rxBuf[0] & MSB(TMP117_EEPROM_EEPROM_BUSY_ENABLE));
}

/*
 *  ======== writeDelay ========
 *  If necessary, wait for a register write to be saved to EEPROM
 */
static void writeDelay(TMP117_Handle sensor)
{
    if (!sensor->nvmCommit) {
        return; /* nothing to do if we're not saving to EEPROM */
    }

    /* Wait delay 7 milliseconds for EEPROM to not be busy */
    mcu_msWait(7);

    /* wait for sensor to not be busy */
    pollBusy(sensor);
}

/*
 *  ======== writeReg ========
 */
static void writeReg(TMP117_Handle sensor, int regAddr, uint16_t value)
{
    uint8_t txBuf[3];

    txBuf[0] = regAddr;
    txBuf[1] = MSB(value);
    txBuf[2] = LSB(value);
    mcu_i2cTransfer(sensor->busId, sensor->devAddr, txBuf, 3, NULL, 0);
    writeDelay(sensor);
}

/*
 *  ======== reset ========
 *  Reset the sensor and wait for it to be ready
 */
static void reset(TMP117_Handle sensor)
{
    uint8_t txBuf[3];

    txBuf[0] = TMP117_CONFIG;
    txBuf[1] = MSB(TMP117_CONFIG_SOFT_RESET_ENABLE);
    txBuf[2] = LSB(TMP117_CONFIG_SOFT_RESET_ENABLE);
    mcu_i2cTransfer(sensor->busId, sensor->devAddr, txBuf, 3, NULL, 0);
    mcu_msWait(2);
}

/*
 *  ======== TMP117_config ========
 *  Configure device with current settings
 *
 *  Note: after a sensor reset, the EEPROM is locked and remains locked for
 *  approximatly 1.5 ms and all register writes are ignored.
 */
void TMP117_config(TMP117_Handle sensor)
{
    /* Ensure sensor is not processing a reset */
    pollBusy(sensor);

    if (sensor->nvmCommit) {
        /* Unlock EEPROM if we're saving registers */
        unlock(sensor);
    }

    /* High Limit Register Write */
    writeReg(sensor, TMP117_THIGH, sensor->thigh);

    /* Low Limit Register Write */
    writeReg(sensor, TMP117_TLOW, sensor->tlow);

    /* Temperature Offset Register Write */
    writeReg(sensor, TMP117_OFFSET_TEMP, sensor->offsetTemp);
    
    if (!sensor->preserveNIST) {
        /* EEPROMx Registers Write */
        writeReg(sensor, TMP117_EEPROM1, sensor->eeprom1);
        writeReg(sensor, TMP117_EEPROM2, sensor->eeprom2);
        writeReg(sensor, TMP117_EEPROM3, sensor->eeprom3);
    }

    /* Configuration Register Write */
    /* Ensure SOFT_RESET isn't set - otherwise CONFIG setting won't work */
    writeReg(sensor, TMP117_CONFIG, sensor->config & (~TMP117_CONFIG_SOFT_RESET_ENABLE));

    /* lock EEPROM to ensure it's no longer writeable */
    if (sensor->config & TMP117_CONFIG_SOFT_RESET_ENABLE) {
        reset(sensor); /* reset also locks EEPROM */
    }
    else {
        lock(sensor);
    }
}

/*
 *  ======== TMP117_read ========
 *  Read raw temperature value
 */
int32_t TMP117_read(TMP117_Handle sensor)
{
    uint8_t txBuf[3];
    uint8_t rxBuf[2] = {0};
    int32_t tmp;

    /* If needed, trigger one shot measurement */
    if (sensor->config & TMP117_CONFIG_CONV_MODE_SD) {
        /* Reset one-shot enable */
        txBuf[0] = TMP117_CONFIG;
        txBuf[1] = MSB(sensor->config | TMP117_OS_ENABLE);
        txBuf[2] = LSB(sensor->config);
        mcu_i2cTransfer(sensor->busId, sensor->devAddr, txBuf, 3, NULL, 0);

        /* Wait for conversion to complete */
        mcu_msWait(sensor->osWait);
    }

    /* Temperature Register Read */
    txBuf[0] = TMP117_TEMP;
    mcu_i2cTransfer(sensor->busId, sensor->devAddr, txBuf, 1, rxBuf, 2);

    /* Sign extend and combine */
    tmp = (((int32_t)(int8_t)rxBuf[0]) << 8) | rxBuf[1];

    return (tmp);
}

/*
 *  ======== TMP117_toIntCelsius ========
 *  Convert raw temperature register value to degrees Celsius rounded to the
 *  nearest integer
 */
int32_t TMP117_toIntCelsius(int32_t x)
{
    /* Optional: Add bias to round before the final truncation */
    x += (x >= 0) ? (1 << 6) : -(1 << 6);

    /* Convert Q7 value to whole number */
    x /= 1 << 7; /* use division so small negative values round to 0 */

    return x;
}

/*
 *  ======== TMP117_toMilliCelsius ========
 *  Convert raw temperature register value to milli-degrees Celsius
 */
int32_t TMP117_toMilliCelsius(int32_t x)
{
    /* Scale to milli-degrees, convert Q7 value to a whole number */
    return ((1000 * x) >> 7);
}

/*
 *  ======== TMP117_toFloatCelsius ========
 *  Convert raw temperature register value to degrees Celsius
 */
float TMP117_toFloatCelsius(int32_t x)
{
    /* Convert Q7 value to a float */
    return ((float)x * 0.0078125f);
}


